/*
 * Copyright (c) 2019-2025 Amazon.com, Inc. or its affiliates.  All rights
 * reserved.
 * Portions Copyright (C) 2018 Sigmastar Technology Corp
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/platform_device.h>

#include <linux/slab.h>
#include <linux/firmware.h>
#include <linux/delay.h>
#include <linux/mmc/card.h>
#include <linux/mmc/sdio_func.h>
#include <linux/mmc/sdio_ids.h>
#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/skbuff.h>
#include <linux/ip.h>

#include <linux/netdevice.h>
#include <linux/gpio.h>

#include "amazon_net.h"
#include "amazon_net_ioctl.h"

#include "mt7682_sdio_reg.h"
#include <linux/rtnetlink.h>

static int disable_comms;
module_param(disable_comms, int, 0644);
MODULE_PARM_DESC(disable_comms, "Disable SDIO comms");

#define MT7682_SDIO_BLOCK_SIZE 256

#define MAX_CHANNELS (32)

#define MAX_TX_COUNT_PER_WORKLOOP 32

#define MIN_PKT_LEN (1)
#define MAX_TX_PKT_BLKS	(12)
#define MAX_RX_PKT_BLKS	(15)

#define MAX_TX_PKT_LEN	(MAX_TX_PKT_BLKS * MT7682_SDIO_BLOCK_SIZE) /* 3072 - Same as the receiving DMA buffers on Red */
#define MAX_RX_PKT_LEN (MAX_RX_PKT_BLKS * MT7682_SDIO_BLOCK_SIZE)

#define RX_BUFFER_SIZE_BYTES	(8192)
#define TX_BUFFER_SIZE_BYTES	(MT7682_SDIO_BLOCK_SIZE * MAX_TX_PKT_BLKS)
#define MIN_MAX_TX_SIZE		(6 * MT7682_SDIO_BLOCK_SIZE)
#define MAX_MAX_TX_SIZE		(TX_BUFFER_SIZE_BYTES - sizeof(struct tx_header))
#define DEFAULT_MAX_TX_LEN	(MAX_MAX_TX_SIZE)
#define MT7686_AGGREGATED_MASK	(0x80)

#define SPIS_CMD_ETH_PKT_TX		0x13

#define TX_COMPLETE_TIMEOUT_MS  (10000)

#define TX_QUEUE_HIGH_THRESHOLD		5	/* Tx queue high water mark */
#define TX_QUEUE_LOW_THRESHOLD		2	/* Tx queue low water mark */

#define SDIO_SWINT_MAILBOX0_RECV      0x00000100

#define SDIO_SWINT_BUSCTL_BUSY_SET    0x00000200
#define SDIO_SWINT_BUSCTL_BUSY_REVOKE 0x00000400

#define SDIO_SWINT_BUSCTL_BUSY_ACK   (4)

#define ATTR_CACHELINE_ALIGNED __aligned(ARCH_DMA_MINALIGN)

enum {
	SDIO_BUS_ACTIVE = 0,
	SDIO_BUS_SUSPENDED = 1,
	SDIO_BUS_STATE_NOT_CHANGED = 2
};

enum {
	SDIO_RQ_ACTIVE = 0,
	SDIO_RQ_SUSPENDED = 1,
	SDIO_RQ_ACTIVE_HIGH_WATERMARK_HIT = 2,
	SDIO_RQ_ACTIVE_HIGH_WATERMARK_RECOVERED = 3,

	SDIO_RQ_LAST
};

enum {
	ICMD_FEATURE_FLAGS = 0,
	ICMD_MAX_HOST_TX_SIZE = 1,
	ICMD_MAX_HOST_RX_SIZE = 2,
	ICMD_CH_AGGREGATION_FLAGS = 3,

	ICMD_LAST
};

enum {
	/* Old version. Should be ignored. */
	SDIO_FF_AGGREGATION = BIT(0),
	/* New version to maintain backwards compatibility */
	SDIO_FF_AGGREGATION_V2 = BIT(1),

	SDIO_FF_LAST
};

/* Define grouped channel priorities. */
#define END_OF_GROUP	((u8)0x80)
static u8 channel_priority[] = {
	/* Housekeeper, Internal, ChiMP control */
	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 | END_OF_GROUP,
	/* ChiMP media */
	16, 17, 11, 12 | END_OF_GROUP,
	/* Snapshots data */
	13, 14, 15, 18, 19, 20 | END_OF_GROUP,
	/* Snapshots control */
	21, 22, 23 | END_OF_GROUP,
	/* Data Transfer Service */
	24, 25, 26, 27, 28, 29, 30, 31 | END_OF_GROUP
};

static const u32 default_msg_level = (NETIF_MSG_DRV   |
				      NETIF_MSG_PROBE |
				      NETIF_MSG_LINK  |
				      NETIF_MSG_IFUP  |
				      NETIF_MSG_IFDOWN);

struct mt7682_amazon_sdio_priv {
	struct amazon_net_priv net;
	struct sdio_func *func;
	u8 busy_det_pin;
	u8 bus_state;

	// queue elements
	struct sk_buff_head	tx_wait_q[MAX_CHANNELS];
	int tx_wait_q_paused[MAX_CHANNELS];
	u32 msg_enable;
	struct work_struct	iot_eth_work;
	struct workqueue_struct *iot_eth_work_queue;
	struct mutex		lock;
	struct completion	tx_done;

	struct timer_list	d2h_handler_timer;
	u16			seq_num;
	unsigned long		flags;
	u32			feature_flags;
	u32			ch_aggregation_flags;
	u32			max_tx_size;
	u32			*rx_buf;
	u32			*tx_buf;

#		define EVENT_D2H_INTR		1
#		define EVENT_TX			2
#		define EVENT_TX_INDIC		3
};

struct tx_header {
	u32  length: 16;
	u32  reserved: 13;
	u32  tx_type: 3;
};

#define AGGREGATED_LEN(X)	\
	((X) - sizeof(struct mt7682_protocol_header) + sizeof(u32))

#define INTERNAL_SDIO_CHANNEL	1

static const struct amazon_internal_command INTERNAL_COMMAND_CONFIG = {
	ICMD_MAX_HOST_RX_SIZE,
	sizeof(uint32_t),
	MAX_RX_PKT_LEN
};

/* Specially crafted packet for signalling SDIO is ready */
static const struct mt7682_protocol_header INDICATION_PACKET_DATA = {
	INTERNAL_SDIO_CHANNEL,			/* Channel number */
	sizeof(INTERNAL_COMMAND_CONFIG),	/* Size of payload */
	0			/* Reserved */
};

#define INDICATION_PACKET_SIZE (sizeof(INDICATION_PACKET_DATA) + \
	sizeof(INTERNAL_COMMAND_CONFIG))

static int sdio_func_wr(struct sdio_func *mt7682_func,
			unsigned int u4register,
			void *pbuffer,
			unsigned int length)
{
	int ret;

	/*addr 4-byte align check*/
	if (WARN((((u32)pbuffer) & 0x03) || !mt7682_func,
		 "%s: Bad arguments", __func__))
		return -EINVAL;

	ret = sdio_writesb(mt7682_func, u4register, pbuffer, length);
	if (ret) {
		dev_err(&mt7682_func->dev,
			"[ERR] function 1 write fail : addr : 0x%x , size : %d, err_ret: 0x%x\n",
			u4register, length, ret);
	}
	return 0;
}

static int sdio_func_rd(struct sdio_func *mt7682_func,
			unsigned int u4register,
			void *pbuffer,
			unsigned int length)
{
	int ret;

	if (WARN(!pbuffer || !mt7682_func, "%s: Bad arguments", __func__))
		return -EINVAL;

	ret = sdio_readsb(mt7682_func, pbuffer, u4register, length);

	if (ret) {
		dev_err(&mt7682_func->dev,
			"[ERR] function 1 read fail : addr : 0x%x , size : %d, err_ret: 0x%x\n",
			u4register, length, ret);
	}

	return ret;
}

static int sdio_receive_pkt(struct sdio_func *mt7682_func, u8 *rx_buf)
{
	u32 total_len = 0;
	u32 wrplr_reg ATTR_CACHELINE_ALIGNED = 0;

	sdio_func_rd(mt7682_func, SDIO_IP_WRPLR, &wrplr_reg, 4);

	total_len = wrplr_reg & 0xffff;

	if (sdio_func_rd(mt7682_func, SDIO_IP_WRDR0, rx_buf, total_len)) {
		dev_err(&mt7682_func->dev,
			"[ERR],%s, sdio_func_rd SDIO_IP_WRDR0 fail\n",
			__func__);
		return -EINVAL;
	}

	return total_len;
}

static int sdio_send_pkt(struct sdio_func *mt7682_func,
			 u8 *tx_buf,
			int data_length)
{
	int ret;

	// Round up to whole block size
	if (data_length > MT7682_SDIO_BLOCK_SIZE)
		data_length = round_up(data_length, MT7682_SDIO_BLOCK_SIZE);

	ret = sdio_func_wr(mt7682_func, SDIO_IP_WTDR1, tx_buf, data_length);
	if (ret) {
		dev_err(&mt7682_func->dev,
			"[ERR] %s => sdio_func_wr 0x%08x len=%d error %d\n",
			__func__,
			SDIO_IP_WTDR1,
			data_length + sizeof(struct tx_header),
			ret);
		return -EINVAL;
	}

	return 0;
}

static int sdio_hif_get_driver_own(struct sdio_func *mt7682_func)
{
	int ret;
	u32 value ATTR_CACHELINE_ALIGNED;
	u32 cnt = 50;

	dev_dbg(&mt7682_func->dev, "[%s]<==========>\n", __func__);

	//Set driver own
	value = W_FW_OWN_REQ_CLR;
	ret = sdio_func_wr(mt7682_func, SDIO_IP_WHLPCR, &value, 4);
	if (ret)
		return ret;

	while (cnt--) {
		ret = sdio_func_rd(mt7682_func, SDIO_IP_WHLPCR, &value, 4);
		if (ret)
			return ret;

		if (value & W_DRV_OWN_STATUS)
			return 0;

		//hal_gpt_delay_ms(50);
	}
	//wait_for_completion_timeout(&sdio_wait_rx_done_event, 1*HZ);

	return -ETIMEDOUT;
}

static int sdio_hif_enable_interrupt(struct sdio_func *mt7682_func)
{
	u32 value ATTR_CACHELINE_ALIGNED;
	int ret;

	value = W_INT_EN_SET;

	ret = sdio_func_wr(mt7682_func, SDIO_IP_WHLPCR, &value, 4);
	if (ret)
		return ret;

	sdio_func_rd(mt7682_func, SDIO_IP_WTSR0, &value, 4);
	sdio_func_rd(mt7682_func, SDIO_IP_WTSR1, &value, 4);

	value = TX_DONE_INT |
		RX0_DONE_INT |
		SDIO_SWINT_MAILBOX0_RECV |
		SDIO_SWINT_BUSCTL_BUSY_SET |
		SDIO_SWINT_BUSCTL_BUSY_REVOKE;

	ret = sdio_func_wr(mt7682_func, SDIO_IP_WHIER, &value, 4);
	if (ret)
		return ret;

	dev_dbg(&mt7682_func->dev,
		"SDIO enabled interrupts : WHIER=0x%08x\n",
		value);

	ret = sdio_func_rd(mt7682_func, SDIO_IP_WHCR, &value, 4);
	if (ret)
		return ret;

	value |= W_INT_CLR_CTRL;

	return sdio_func_wr(mt7682_func, SDIO_IP_WHCR, &value, 4);
}

static int sdio_hif_disable_interrupt(struct sdio_func *mt7682_func)
{
	u32 value ATTR_CACHELINE_ALIGNED;
	int ret;

	value = W_INT_EN_CLR;

	ret = sdio_func_wr(mt7682_func, SDIO_IP_WHLPCR, &value, 4);
	if (ret)
		return ret;

	value = 0x00000080;

	return sdio_func_wr(mt7682_func, SDIO_IP_WHIER, &value, 4);
}

static void
process_channel_aggregation_flags(struct mt7682_amazon_sdio_priv *priv,
				  u8 *data, int len)
{
	if (len != sizeof(u32)) {
		netdev_warn(priv->net.ndev,
			    "%s: invalid length: %d\n",
			    __func__,
			    len);
		return;
	}

	priv->ch_aggregation_flags = *(u32 *)data;
	trace_printk("channel aggregation flags: 0x%08x\n", priv->ch_aggregation_flags);
}

static void process_feature_flags(struct mt7682_amazon_sdio_priv *priv,
				 u8 *data, int len)
{
	if (len != sizeof(u32)) {
		netdev_warn(priv->net.ndev,
			    "%s: invalid length: %d\n",
			    __func__,
			    len);
		return;
	}

	priv->feature_flags = *(u32 *)data;
	netdev_dbg(priv->net.ndev,
		    "%s: feature flags: 0x%08x\n",
		    __func__,
		    priv->feature_flags);
}

static void process_max_tx_size(struct mt7682_amazon_sdio_priv *priv,
			       u8 *data, int len)
{
	u32 max_tx_size;

	if (len != sizeof(u32)) {
		netdev_warn(priv->net.ndev,
			    "%s: invalid length: %d\n",
			    __func__,
			    len);
		return;
	}

	max_tx_size = *(u32 *)data;

	/* Ensure the value is within the supported range */
	if (max_tx_size > MAX_MAX_TX_SIZE) {
		/* Would be too big for the tx buffer */
		max_tx_size = MAX_MAX_TX_SIZE;
	} else if (max_tx_size < MIN_MAX_TX_SIZE) {
		/* Would be too small for the largest single packet */
		max_tx_size = MIN_MAX_TX_SIZE;
	}

	priv->max_tx_size = max_tx_size;
	netdev_dbg(priv->net.ndev,
		    "%s: max tx size: %d\n",
		    __func__,
		    max_tx_size);
}

static void process_internal_pkt(struct mt7682_amazon_sdio_priv *priv,
				 u8 *data, int len)
{
	while (len > 1) {
		u8 cmd = *data++;
		u8 cmd_len = *data++;

		switch (cmd) {
		case ICMD_FEATURE_FLAGS:
			process_feature_flags(priv, data, cmd_len);
			break;
		case ICMD_MAX_HOST_TX_SIZE:
			process_max_tx_size(priv, data, cmd_len);
			break;
		case ICMD_CH_AGGREGATION_FLAGS:
			process_channel_aggregation_flags(priv, data, cmd_len);
			break;
		default:
			netdev_dbg(priv->net.ndev,
				   "%s: invalid internal command: %d, %d",
				   __func__,
				   cmd,
				   cmd_len);
			break;
		}

		data += cmd_len;
		len -= 2 + cmd_len;	/* command + len + payload */
	}
}

static int mt7682_send_eth_pkt(struct mt7682_amazon_sdio_priv *priv,
			       struct sk_buff_head *tx_skb_list)
{
	struct sdio_func *mt7682_func = priv->func;
	int status;
	struct sk_buff *tx_skb;
	u8 *tx_buffer;
	struct tx_header *tx_buffer_header;
	int sendlen;
	int tx_q_len = skb_queue_len(tx_skb_list);
	static u32 last_len;

	void *data;
	u16 len = 0;
	struct mt7682_protocol_header hdr;

	if (!tx_q_len) {
		dev_dbg(&mt7682_func->dev, "nothing to send");
		return 0;
	}

	data = skb_peek(tx_skb_list)->data;
	hdr = *(struct mt7682_protocol_header *)data;

	if (tx_q_len == 1) {
		/* Normal case, single buffer packet */
		len = skb_peek(tx_skb_list)->len;
	} else {
		/* Calculate aggregated length */
		len = sizeof(struct mt7682_protocol_header);
		skb_queue_walk(tx_skb_list, tx_skb) {
			dev_dbg(&mt7682_func->dev, "add %d byte buffer",
				tx_skb->len);
			len += ALIGN(AGGREGATED_LEN(tx_skb->len), 4);
		}
	}

	tx_buffer_header = (struct tx_header *)priv->tx_buf;
	tx_buffer_header->length = len + sizeof(struct tx_header);
	tx_buffer = (u8 *)(tx_buffer_header + 1);

	sendlen = tx_buffer_header->length;
	if (sendlen > MT7682_SDIO_BLOCK_SIZE)
		sendlen = ALIGN(sendlen, MT7682_SDIO_BLOCK_SIZE);

	if (sendlen > TX_BUFFER_SIZE_BYTES) {
		dev_err(&mt7682_func->dev,
			"too much data to send: %u > %u",
			tx_buffer_header->length, TX_BUFFER_SIZE_BYTES);
		return -ENOMEM;
	}

	/* Clear previously used part of buffer */
	if (last_len > len)
		memset(tx_buffer + len, 0, last_len - len);
	last_len = len;

	if (tx_q_len == 1) {
		memcpy(tx_buffer, data, len);
	} else {
		u8 *dst = tx_buffer;

		dev_dbg(&mt7682_func->dev, "aggregated %d buffers", tx_q_len);
		/* Adjust first buffer's header for the aggregated size and
		 * put that at the start of the buffer
		 */
		hdr.channel |= MT7686_AGGREGATED_MASK;
		hdr.size = len - sizeof(struct mt7682_protocol_header);
		*(struct mt7682_protocol_header *)dst = hdr;
		dst += sizeof(struct mt7682_protocol_header);

		skb_queue_walk(tx_skb_list, tx_skb) {
			/* Insert sub-packet length and copy data */
			u32 sub_pkt_len = tx_skb->len -
					  sizeof(struct mt7682_protocol_header);
			u8 padding = ALIGN(sub_pkt_len, 4) - sub_pkt_len;
			*(u32 *)dst = sub_pkt_len;
			dst += sizeof(sub_pkt_len);
			data = tx_skb->data +
			       sizeof(struct mt7682_protocol_header);
			memcpy(dst, data, sub_pkt_len);
			dst += sub_pkt_len;
			if (padding) {
				memset(dst, 0, padding);
				dst += padding;
			}
		}
	}

	dev_dbg(&mt7682_func->dev, "%s (%d len)\n", __func__, len);
#if defined(CONFIG_DYNAMIC_DEBUG) || defined(DEBUG)
	{
		char buff[300];

		escape_string_buffer(buff,
				     sizeof(buff),
				     (u8 *)priv->tx_buf,
				     sendlen);
		dev_vdbg(&mt7682_func->dev, "%s", buff);
	}
#endif // CONFIG_DYNAMIC_DEBUG || DEBUG

	dev_vdbg(&mt7682_func->dev,
		 "tx_skb_data=%p l1=%d l2=%d, mt7682_func=%p\n",
		 priv->tx_buf, len, sendlen, mt7682_func);

	do {
		sdio_claim_host(mt7682_func);
		status = sdio_send_pkt(mt7682_func,
				       (u8 *)priv->tx_buf,
				       sendlen);
		sdio_release_host(mt7682_func);

		if (!status)
			break;

		if (status != -ETIME) {
			dev_err(&mt7682_func->dev,
				"%s error %d\n",
				__func__,
				status);

			break;
		}

		dev_err(&mt7682_func->dev,
			"%s timeout %d, retry\n",
			__func__, status);
	} while (true);

	if (status) {
		dev_err(&mt7682_func->dev,
			"sdio host send error! %d\n", status);
		return status;
	}
	dev_dbg(&mt7682_func->dev,
		"sdio host send ok. (%d bytes - %d bytes padded)\n",
		len, sendlen);
	return 0;
}

static int iot_eth_rx(struct net_device *dev)
{
	struct mt7682_amazon_sdio_priv *host =
		(struct mt7682_amazon_sdio_priv *)netdev_priv(dev);

	struct sk_buff *skb;
	int rxlen = -1;

	netdev_dbg(dev, "%s: func=%p\n", __func__, host->func);

	rxlen = sdio_receive_pkt(host->func, (u8 *)host->rx_buf);
	if (rxlen < 0) {
		netdev_err(dev, "SDIO host receive error!\n");
		return rxlen;
	}

	netdev_dbg(dev, "SDIO RX: %d bytes\n", rxlen);
#if defined(CONFIG_DYNAMIC_DEBUG) || defined(DEBUG)
	{
		char buff[300];

		escape_string_buffer(buff,
				     sizeof(buff),
				     (u8 *)host->rx_buf,
				     rxlen);
		netdev_vdbg(dev, "%s", buff);
	}
#endif // CONFIG_DYNAMIC_DEBUG || DEBUG

	do {
		struct mt7682_protocol_header *header =
			(struct mt7682_protocol_header *)host->rx_buf;
		u16 real_len = header->size + sizeof(*header);

		if (header->channel == INTERNAL_SDIO_CHANNEL) {
			if (real_len == rxlen) {
				process_internal_pkt(host, (u8 *)(header + 1),
						     header->size);
			}
			break;
		}

		if (!netif_running(dev)) {
			netdev_info(dev, "rx pkt dropped!\n");
			break;
		}

		skb = netdev_alloc_skb(dev, real_len);
		if (skb) {
			u8 *pread;

			pread = (u8 *)skb_put(skb, real_len);

			memcpy_fromio(pread, host->rx_buf, real_len);

			/* Packet Status check */
			if (rxlen < MIN_PKT_LEN) {
				if (rxlen)
					netdev_warn(dev,
						    "RX: Bad Packet (runt). len = %d\n",
						    rxlen);
				dev_kfree_skb(skb);
				break;
			}

			dev->stats.rx_bytes += rxlen;

			/* Pass to upper layer */
			amazon_net_receive(host->net.ndev, skb);

			dev->stats.rx_packets++;
		}
	} while (0);

	return 0;
}

static bool
h2d_receive_mailbox(struct sdio_func *func, uint32_t *cmd)
{
	return (sdio_func_rd(func, SDIO_IP_D2HRM0R, cmd, 4) == 0);
}

static bool h2d_sw_interrupt(struct sdio_func *func, uint8_t intr)
{
	u32 val ATTR_CACHELINE_ALIGNED = 1 << (intr + 16);

	return (sdio_func_wr(func, SDIO_IP_WSICR, &val, 4) == 0);
}

static void send_indication_packet(struct mt7682_amazon_sdio_priv *priv)
{
	struct sk_buff *skb;
	u8 *ptr;
	struct sk_buff_head tx_skb_list;
	int ret = 0;

	skb = netdev_alloc_skb(priv->net.ndev, INDICATION_PACKET_SIZE);
	if (!skb) {
		netdev_err(priv->net.ndev, "%s: alloc failed\n", __func__);
		return;
	}

	ptr = skb_put(skb, INDICATION_PACKET_SIZE);
	memcpy(ptr, &INDICATION_PACKET_DATA, sizeof(INDICATION_PACKET_DATA));
	ptr += sizeof(INDICATION_PACKET_DATA);
	memcpy(ptr, &INTERNAL_COMMAND_CONFIG, sizeof(INTERNAL_COMMAND_CONFIG));

	__skb_queue_head_init(&tx_skb_list);
	__skb_queue_tail(&tx_skb_list, skb);

	netdev_warn(priv->net.ndev,
		   "%s: sending indication packet\n",
		   __func__);

	ret = mt7682_send_eth_pkt(priv, &tx_skb_list);
	__skb_queue_purge(&tx_skb_list);

	/* Wait for transmit complete interrupt */
	if (!ret) {
		unsigned long timeout = msecs_to_jiffies(TX_COMPLETE_TIMEOUT_MS);

		priv->net.ndev->stats.tx_bytes += INDICATION_PACKET_SIZE;
		priv->net.ndev->stats.tx_packets++;

		ret = wait_for_completion_timeout(&priv->tx_done, timeout);
		if (!ret)
			netdev_err(priv->net.ndev, "%s: TX error! (%d)\n",
				__func__, ret);
	}
}

/*******************************************************************/
/* SDIO callbacks                                                  */
/*******************************************************************/

void mt7682_sdio_interrupt(struct sdio_func *func)
{
	u32 whisr ATTR_CACHELINE_ALIGNED;
	struct mt7682_amazon_sdio_priv *host = sdio_get_drvdata(func);
	struct net_device *ndev = host->net.ndev;
	u8 bus_busy_state = SDIO_BUS_STATE_NOT_CHANGED;
	int kick_work_queue = 0;

	dev_dbg(&func->dev, "############%s##############\n", __func__);

	sdio_claim_host(func);
	sdio_func_rd(func, SDIO_IP_WHISR, &whisr, 4);

	dev_vdbg(&func->dev,
		 "###%s: whisr = 0x%x\n",
		 __func__, whisr);

	if (whisr & TX_DONE_INT) {
		u32 wtsr0 ATTR_CACHELINE_ALIGNED;

		sdio_func_rd(func, SDIO_IP_WTSR0, &wtsr0, 4);

		complete(&host->tx_done);
	}

	if (whisr & RX0_DONE_INT) {
		dev_vdbg(&func->dev, "###RX#0 DONE!\n");
		iot_eth_rx(ndev);
	}

	if (whisr & SDIO_SWINT_BUSCTL_BUSY_SET)
		bus_busy_state = SDIO_BUS_SUSPENDED;
	else if (whisr & SDIO_SWINT_BUSCTL_BUSY_REVOKE)
		bus_busy_state = SDIO_BUS_ACTIVE;

	if (bus_busy_state != SDIO_BUS_STATE_NOT_CHANGED) {
		host->bus_state = bus_busy_state;
		h2d_sw_interrupt(func, SDIO_SWINT_BUSCTL_BUSY_ACK);
		dev_dbg(&func->dev, "Bus busy state: %d\n", bus_busy_state);
		kick_work_queue |= bus_busy_state == SDIO_BUS_ACTIVE;
	}

	// Clear sw interrupts
	if (whisr & 0xffffff00) {
		sdio_func_wr(func, SDIO_IP_WHISR, &whisr, 4);
	}

	if (whisr & SDIO_SWINT_MAILBOX0_RECV) {
		u8 channel;
		u32 mail_value ATTR_CACHELINE_ALIGNED = 0;

		h2d_receive_mailbox(func, &mail_value);

		for (channel = 0;  channel < 32; channel++) {
			bool new_state  = mail_value & 0x1;
			bool curr_state = host->tx_wait_q_paused[channel];

			if (new_state != curr_state) {
				if (new_state == 0) {
					// Resuming transmission, check work queue
					dev_dbg(&func->dev, "ch%d resumed %d\n",
						channel,
						host->tx_wait_q_paused[channel]);
					kick_work_queue = 1;
				} else {
					dev_dbg(&func->dev, "ch%d paused %d\n",
						channel,
						host->tx_wait_q_paused[channel]);
				}

				host->tx_wait_q_paused[channel] = new_state;
			}
			mail_value = mail_value >> 1;
		}
	}

	if (whisr & 0x40) {
		u32 wasr ATTR_CACHELINE_ALIGNED;

		sdio_func_rd(func, SDIO_IP_WASR, &wasr, 4);

		dev_info(&func->dev,
			 "###%s: wasr = x%x\n",
			 __func__, wasr);

		if (wasr & RX0_UNDERFLOW)
			dev_info(&func->dev, "RX0_UNDERFLOW");

		if (wasr & TX1_OVERFLOW)
			dev_info(&func->dev, "TX1_OVERFLOW");

		if (wasr & TX0_OVERFLOW)
			dev_info(&func->dev, "TX0_OVERFLOW");

		if (wasr & FW_OWN_INVALID_ACCESS)
			dev_info(&func->dev, "FW_OWN_INVALID_ACCESS");
	}
	if (kick_work_queue) {
		set_bit(EVENT_TX, &host->flags);
		queue_work(host->iot_eth_work_queue,
			   &host->iot_eth_work);
	}
	sdio_release_host(func);
}

static int mt7682_init(struct sdio_func *func)
{
	int ret;

	sdio_claim_host(func);

	ret = sdio_enable_func(func);
	if (ret) {
		dev_err(&func->dev,
			"mt7682_sdio: Failed to enable F1 Err: 0x%08x",
			ret);
		goto release;
	}

	ret = sdio_set_block_size(func, MT7682_SDIO_BLOCK_SIZE);
	if (ret)
		goto disable;

	ret = sdio_hif_get_driver_own(func);
	dev_info(&func->dev, "sdio_hif_get_driver_own: ret = %d\n", ret);
	if (ret)
		goto disable;

	ret = sdio_hif_enable_interrupt(func);

	func->cur_blksize = MT7682_SDIO_BLOCK_SIZE;
	ret = sdio_claim_irq(func, mt7682_sdio_interrupt);
	if (ret)
		goto release_irq;

	sdio_release_host(func);

	return 0;

release_irq:
	sdio_release_irq(func);
disable:
	sdio_disable_func(func);
release:
	sdio_release_host(func);
	return ret;
}

static int mt7682_close(struct sdio_func *func)
{
	sdio_claim_host(func);
	sdio_hif_disable_interrupt(func);
	sdio_release_irq(func);
	sdio_release_host(func);
	return 0;
}

static int iot_eth_hard_xmit(struct mt7682_amazon_sdio_priv *host,
			     unsigned char channel)
{
	struct sk_buff_head *tx_wait_q = &host->tx_wait_q[channel];
	struct sk_buff_head tx_skb_list;
	int byte_cnt = sizeof(struct mt7682_protocol_header);
	int ret = 0;

	/* Local queue of buffers to send in one go */
	__skb_queue_head_init(&tx_skb_list);

	do {
		struct sk_buff *tx_skb;

		tx_skb = skb_peek(tx_wait_q);
		if (!tx_skb) {
			/* No more buffers, stop */
			if (!skb_queue_len(&tx_skb_list))
				ret = -ENOMEM;
			break;
		}

		if (tx_skb->len > MAX_TX_PKT_LEN) {
			/* Oversize buffer, stop */
			if (!skb_queue_len(&tx_skb_list)) {
				/* No previous buffers, fail */
				netdev_err(host->net.ndev,
					   "oversize buffer, %d > %d\n",
					   tx_skb->len,
					   MAX_TX_PKT_LEN);
				skb_unlink(tx_skb, tx_wait_q);
				dev_kfree_skb(tx_skb);
				ret = -EINVAL;
			}
			break;
		}

		if (byte_cnt + ALIGN(AGGREGATED_LEN(tx_skb->len), 4) >
				host->max_tx_size) {
			/* Too much for one packet, stop, leave buf in queue */
			break;
		}

		netdev_dbg(host->net.ndev,
			   "%s: ch %d, %d bytes\n",
			   __func__,
			   channel,
			   tx_skb->len);

		/* Move buffer to tx_skb_list for tx */
		skb_unlink(tx_skb, tx_wait_q);
		__skb_queue_tail(&tx_skb_list, tx_skb);
		byte_cnt += ALIGN(AGGREGATED_LEN(tx_skb->len), 4);

		if (!(host->feature_flags & SDIO_FF_AGGREGATION_V2) || !(host->ch_aggregation_flags & BIT(channel))) {
			/* Aggregation not supported, send packets singly */
			break;
		}
	} while (true);

	if (!ret) {
		/* Single buffers are sent unaggregated, adjust */
		if (skb_queue_len(&tx_skb_list) == 1)
			byte_cnt -= sizeof(u32);
		else {
			netdev_dbg(host->net.ndev,
				   "%s: ch %d, %d packets aggregated\n",
				   __func__,
				   channel,
				   skb_queue_len(&tx_skb_list));
		}
		ret = mt7682_send_eth_pkt(host, &tx_skb_list);
	}

	__skb_queue_purge(&tx_skb_list);

	if (!ret) {
		unsigned long timeout;

		host->net.ndev->stats.tx_bytes += byte_cnt;

		host->net.ndev->stats.tx_packets++;

		/* Wait for transmit complete interrupt */
		timeout = msecs_to_jiffies(TX_COMPLETE_TIMEOUT_MS);
		ret = wait_for_completion_timeout(&host->tx_done, timeout);
		if (!ret)
			netdev_err(host->net.ndev,
				   "%s: TX error! (%d)\n",
				   __func__, ret);
	}

	return ret;
}

static void iot_eth_work(struct work_struct *work)
{
	struct mt7682_amazon_sdio_priv *host =
		container_of(work,
			     struct mt7682_amazon_sdio_priv,
			     iot_eth_work);

	mutex_lock(&host->lock);

	if (test_and_clear_bit(EVENT_D2H_INTR, &host->flags)) {
		// while(gpio_get_value(MMPF_PIO_REG_GPIO117)==1)
		mt7682_sdio_interrupt(host->func);
			// iot_eth_rx(host->net.ndev);

		// enable_irq (host->net.ndev->irq);
	}

	if (test_and_clear_bit(EVENT_TX_INDIC, &host->flags)) {
		send_indication_packet(host);
	}

	if (test_and_clear_bit(EVENT_TX, &host->flags)) {
		int ret;
		int tx_count = 0;
		int ch_pri = 0;
		int channel;
		bool packet_sent = false;
		u8 last_priority;

		/* TODO: This needs to be processed
		 *       if flow control state changes
		 */

		do {
			if (test_and_clear_bit(EVENT_D2H_INTR, &host->flags)) {
				mt7682_sdio_interrupt(host->func);
			}
			channel = channel_priority[ch_pri] & ~END_OF_GROUP;
			if ((!host->tx_wait_q_paused[channel]) &&
			    (skb_queue_len(&host->tx_wait_q[channel])) &&
			    (host->bus_state == SDIO_BUS_ACTIVE)) {
				ret = iot_eth_hard_xmit(host, channel);
				if (ret)
					break;

				if (++tx_count >= MAX_TX_COUNT_PER_WORKLOOP)
					break;

				packet_sent = true;
			}

			/* Next priority channel */
			last_priority = channel_priority[ch_pri++];
			if (packet_sent && (last_priority & END_OF_GROUP)) {
				/* Restart at highest priority */
				ch_pri = 0;
				packet_sent = false;
			}
		} while (ch_pri < ARRAY_SIZE(channel_priority));

		if (ch_pri < ARRAY_SIZE(channel_priority)) {
			set_bit(EVENT_TX, &host->flags);
			queue_work(host->iot_eth_work_queue,
				   &host->iot_eth_work);
		}
	}

	mutex_unlock(&host->lock);
}

static void iot_eth_d2h_timer_handler(unsigned long data)
{
	struct mt7682_amazon_sdio_priv *host =
		(struct mt7682_amazon_sdio_priv *)data;

// PROBLEM!!!!!!!!!!!!!!!!!!
// ONLY RECEIVING PACKETS WHEN THIS IS ENABLED

	set_bit(EVENT_D2H_INTR, &host->flags);
	queue_work(host->iot_eth_work_queue, &host->iot_eth_work);

	mod_timer(&host->d2h_handler_timer, jiffies + msecs_to_jiffies(100));
}

static u16 amzon_sdio_select_queue(struct net_device *dev,
				   struct sk_buff *skb,
				   void *accel_priv,
				   select_queue_fallback_t fallback)
{
	return 0;
}

static int amazon_sdio_xmit(struct net_device *ndev, struct sk_buff *skb)
{
	struct mt7682_amazon_sdio_priv *priv =
		(struct mt7682_amazon_sdio_priv *)netdev_priv(ndev);

	u8 channel = ((u8 *)skb->data)[0];
	u16 len = skb->len;

	if (channel >= MAX_CHANNELS) {
		netdev_err(ndev, "invalid channel: %d >= %d\n",
		       channel, MAX_CHANNELS);
		return NETDEV_TX_BUSY;
	}

	skb_queue_tail(&priv->tx_wait_q[channel], skb);

	netdev_dbg(priv->net.ndev,
		   "%s: queued %d bytes to ch %d, level %d\n",
		   __func__,
		   len,
		   channel,
		   skb_queue_len(&priv->tx_wait_q[channel]));

	if (!skb->xmit_more || skb_queue_len(&priv->tx_wait_q[channel]) > 4) {
		set_bit(EVENT_TX, &priv->flags);
		queue_work(priv->iot_eth_work_queue, &priv->iot_eth_work);
	}

	return NETDEV_TX_OK;
}

static int amazon_sdio_probe(struct sdio_func *func,
			     const struct sdio_device_id *id)
{
	int ret = -1;
	struct mt7682_amazon_sdio_priv *priv;
	struct net_device *dev_amazon_sdio;
	int channel;

	if (disable_comms) {
		dev_info(&func->dev, "%s: sdio is disabled\n", __func__);
		return -EPERM;
	}

	dev_info(&func->dev,
		 "%s: id=%04x:%04x\n",
		__func__,
		id->vendor,
		id->device);

	dev_amazon_sdio =
		amazon_net_ndev_alloc(sizeof(struct mt7682_amazon_sdio_priv),
				      "amazon_sdio%d");
	if (!dev_amazon_sdio)
		return -ENOMEM;

	priv = (struct mt7682_amazon_sdio_priv *)netdev_priv(dev_amazon_sdio);

	/* The buf allocated by kmalloc will align with ARCH_DMA_MINALIGN */
	priv->rx_buf = kmalloc(RX_BUFFER_SIZE_BYTES, GFP_KERNEL);
	if (!priv->rx_buf)
		goto rx_buf_init_fail;

	priv->tx_buf = kmalloc(TX_BUFFER_SIZE_BYTES, GFP_KERNEL);
	if (!priv->tx_buf)
		goto tx_buf_init_fail;

	memset(priv->rx_buf, 0, RX_BUFFER_SIZE_BYTES);
	memset(priv->tx_buf, 0, TX_BUFFER_SIZE_BYTES);

	mutex_init(&priv->lock);
	init_completion(&priv->tx_done);

	priv->msg_enable = netif_msg_init(-1, default_msg_level);

	for (channel = 0; channel < MAX_CHANNELS; channel++) {
		skb_queue_head_init(&priv->tx_wait_q[channel]);
		priv->tx_wait_q_paused[channel] = 0;
	}
	priv->bus_state = SDIO_BUS_ACTIVE;

	INIT_WORK(&priv->iot_eth_work, iot_eth_work);
	priv->iot_eth_work_queue =
		alloc_ordered_workqueue("%s", WQ_MEM_RECLAIM | WQ_HIGHPRI |
					WQ_CPU_INTENSIVE, "iot_eth_work");
	setup_timer(&priv->d2h_handler_timer, iot_eth_d2h_timer_handler,
		    (unsigned long)priv);

	mod_timer(&priv->d2h_handler_timer, jiffies + msecs_to_jiffies(100));

	priv->func = func;
	priv->max_tx_size = DEFAULT_MAX_TX_LEN;

	sdio_set_drvdata(func, priv);

	mt7682_init(func);

	ret = gpio_request(priv->busy_det_pin, "busy detect pin");
	if (ret) {
		netdev_err(dev_amazon_sdio,
			   "Failed requesting gpio_ro %d\n",
			   priv->busy_det_pin);
		goto resource_init_fail;
	}

//	MHal_GPIO_Pull_Low(priv->busy_det_pin); // GPIO Power Off

	/* Trigger the TX work queue to send the indication packet */
	set_bit(EVENT_TX_INDIC, &priv->flags);
	queue_work(priv->iot_eth_work_queue, &priv->iot_eth_work);

	rtnl_lock();
	ret = amazon_net_ndev_register(dev_amazon_sdio, &amazon_sdio_ops);
	if (ret < 0) {
		netdev_err(dev_amazon_sdio,
			   "failed to register %s\n",
			   dev_amazon_sdio->name);
		rtnl_unlock();
		goto resource_init_fail;
	}

	ret = dev_open(dev_amazon_sdio);
	if (ret) {
		netdev_err(dev_amazon_sdio,
			   "failed to open %s\n",
			   dev_amazon_sdio->name);
		rtnl_unlock();
		goto register_ndev_fail;
	}
	rtnl_unlock();
	return 0;

register_ndev_fail:
	amazon_net_ndev_unregister(dev_amazon_sdio);
resource_init_fail:
	destroy_workqueue(priv->iot_eth_work_queue);
	kfree(priv->tx_buf);
tx_buf_init_fail:
	kfree(priv->rx_buf);
rx_buf_init_fail:
	amazon_net_ndev_free(dev_amazon_sdio);
	return ret;
}

void amazon_sdio_flush_channel(struct net_device *dev, uint8_t channel)
{
	struct sk_buff *skb;
	struct mt7682_amazon_sdio_priv *host =
		(struct mt7682_amazon_sdio_priv *)netdev_priv(dev);

	mutex_lock(&host->lock);
	while (host->tx_wait_q[channel].qlen) {
		skb = skb_dequeue(&host->tx_wait_q[channel]);
		kfree_skb(skb);
	}
	mutex_unlock(&host->lock);
}

static void amazon_sdio_remove(struct sdio_func *func)
{
	struct mt7682_amazon_sdio_priv *host = sdio_get_drvdata(func);
	int channel;

	kfree(host->tx_buf);
	kfree(host->rx_buf);

	sdio_claim_host(func);
	sdio_release_irq(func);
	sdio_release_host(func);

	mt7682_close(func);

	for (channel = 0; channel < MAX_CHANNELS; channel++) {
		amazon_sdio_flush_channel(host->net.ndev, channel);
	}

	destroy_workqueue(host->iot_eth_work_queue);

	amazon_net_ndev_delete(host->net.ndev);
}

/* devices we support, null terminated */
static const struct sdio_device_id amazon_sdmmc_ids[] = {
	{SDIO_DEVICE(SDIO_VENDOR_ID_MEDIATEK, SDIO_DEVICE_ID_MEDIATEK_7686)},
	{SDIO_DEVICE(SDIO_VENDOR_ID_MEDIATEK, SDIO_DEVICE_ID_MEDIATEK_7682)},
	{ /* end: all zeroes */ }
};
MODULE_DEVICE_TABLE(sdio, amazon_sdmmc_ids);

static struct sdio_driver mt7682_amazon_sdio_driver = {
	.probe = amazon_sdio_probe,
	.remove = amazon_sdio_remove,
	.name = DRV_NAME,
	.id_table = amazon_sdmmc_ids,
	.drv = {
		.owner = THIS_MODULE,
	},
};

int amazon_sdio_init(void)
{
	return sdio_register_driver(&mt7682_amazon_sdio_driver);
}

void amazon_sdio_deinit(void)
{
	sdio_unregister_driver(&mt7682_amazon_sdio_driver);
}

static int amazon_sdio_ioctl(struct net_device *dev,
			     struct ifreq *ifreq,
			     int cmd)
{
	struct if_amazon_sdio_req *rq = (struct if_amazon_sdio_req *)ifreq;
	struct mt7682_amazon_sdio_priv *priv =
		(struct mt7682_amazon_sdio_priv *)netdev_priv(dev);
	int ret;

	if (!netif_running(dev))
		return -EINVAL;

	switch (cmd) {
	case SIOCGETQUEUEDBYTECOUNT:
		{
			unsigned int byte_count = 0;
			struct sk_buff *skb;
			unsigned char channel =
				rq->ifr_ifru.ifru_queued_count.channel;
			const struct sk_buff_head *list =
				&priv->tx_wait_q[channel];

			mutex_lock(&priv->lock);
			skb = skb_peek(list);
			while (skb) {
				byte_count += skb->len;
				skb = skb_peek_next(skb, list);
			}
			mutex_unlock(&priv->lock);

			rq->ifr_ifru.ifru_queued_count.count = byte_count;
			ret = 0;
		}
		break;

	case SIOCGETQUEUEDPACKETCOUNT:
		{
			unsigned int n;
			unsigned char channel =
				rq->ifr_ifru.ifru_queued_count.channel;
			const struct sk_buff_head *list =
				&priv->tx_wait_q[channel];

			mutex_lock(&priv->lock);
			n = skb_queue_len(list);
			mutex_unlock(&priv->lock);

			netdev_dbg(dev,
				   "%s: SIOCGETQUEUEDPACKETCOUNT = %d\n",
				   __func__,
				   n);

			rq->ifr_ifru.ifru_queued_count.count = n;
			ret = 0;
		}
		break;

	default:
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}

const struct amazon_net_ops amazon_sdio_ops = {
		.xmit = amazon_sdio_xmit,
		.init = amazon_sdio_init,
		.deinit = amazon_sdio_deinit,
		.select_queue = amzon_sdio_select_queue,
		.do_ioctl = amazon_sdio_ioctl,
		.flush_channel = amazon_sdio_flush_channel,
};
